<?php

namespace App\Controllers;

use App\Models\CarouselModel;
use App\Models\BlogModel;
use App\Models\EventModel;
use App\Models\MemberModel;
use App\Models\UserModel;
use App\Models\GalleryModel;
use App\Models\ProfileModel;

class Admin extends BaseController
{
    protected $carouselModel;
    protected $blogModel;
    protected $eventModel;
    protected $memberModel;
    protected $userModel;
    protected $galleryModel;
    protected $profileModel;


    public function __construct()
    {
        helper(['form', 'url']);
        session();
        $this->carouselModel = new CarouselModel();
        $this->blogModel = new BlogModel();
        $this->eventModel = new EventModel();
        $this->memberModel = new MemberModel();
        $this->userModel = new UserModel();
        $this->galleryModel = new GalleryModel();
        $this->profileModel = new \App\Models\ProfileModel();
    }

    public function index()
    {
        helper('text');

        $search = $this->request->getGet('search');
        $status = $this->request->getGet('status');
        $searchDate = $this->request->getGet('searchDate');

        $query = $this->carouselModel;

        if ($search) {
            $query = $query->groupStart()
            ->like('title', $search)
            ->orLike('sub-title', $search)
            ->groupEnd();
        }
    
        if ($status && in_array($status, ['Publish', 'Unpublish'])) {
            $query = $query->where('status', ($status)); 
        }

        if ($searchDate) {
            $query = $query->where('date', $searchDate);
        }

        $query = $query->where('deleted_at', null); 

        $data['slides'] = $query->orderBy('created_at', 'DESC')->findAll();

        return view('admin/carousel/index', $data);
    }

    public function list()
    {

        $request = service('request');

        $columns = $request->getGet('columns') ?? [];
        $searchTitle = $columns[2]['search']['value'] ?? '';
        $searchSubtitle = $columns[3]['search']['value'] ?? '';
        $searchDescription = $columns[5]['search']['value'] ?? '';
        $searchDate = $request->getGet('searchDate') ?? '';
        $status = $request->getGet('status') ?? '';
        $page = (int) $request->getGet('page') ?: 1;
        $length = (int) $request->getGet('length') ?: 10;
        $start = ($page - 1) * $length;

        $model = new \App\Models\CarouselModel();
        $builder = $model->builder()->where('deleted_at', null); 

        // Apply filters
        if ($searchTitle) {
            $builder->like('title', $searchTitle);
        }

        if ($searchSubtitle) {
            $builder->like('sub-title', $searchSubtitle);
        }

        if ($searchDescription) {
            $builder->like('description', $searchDescription);
        }

        if ($searchDate) {
            $builder->where('DATE(created_at)', $searchDate);
        }

        if ($status) {
            $builder->where('status', $status);
        }

        $filteredBuilder = clone $builder;
        $filteredRecords = $filteredBuilder->countAllResults();

        // Handle sorting
        $orderColumnIndex = $request->getGet('order')[0]['column'] ?? 4;
        $orderDir = $request->getGet('order')[0]['dir'] ?? 'desc';
        $columnsMap = [
            2 => 'title',
            3 => 'sub-title',
            4 => 'created_at',
            5 => 'description',
        ];
        $orderColumn = $columnsMap[$orderColumnIndex] ?? 'created_at';
        $builder->orderBy($orderColumn, $orderDir);

        // Paginate and fetch
        $slides = $builder->limit($length, $start)->get()->getResultArray();

        // Count all non-deleted records
        $totalRecords = (new \App\Models\CarouselModel())
            ->where('deleted_at', null)
            ->countAllResults();
        $data = [];
        foreach ($slides as $slide) 
        {
            $dataAttributes = 'data-id="' . $slide['sno'] . '"
                data-title="' . htmlspecialchars($slide['title'], ENT_QUOTES) . '"
                data-subtitle="' . htmlspecialchars($slide['sub-title'], ENT_QUOTES) . '"
                data-description="' . htmlspecialchars($slide['description'], ENT_QUOTES) . '"
                data-status="' . $slide['status'] . '"';

            $statusButton = '<form action="' . base_url('admin/toggleStatus/carousel/' . $slide['sno']) . '" method="post" style="display:inline;">
                <button type="submit" class="btn w-80" style="background-color: ' . ($slide['status'] === 'Publish' ? '#32CD32' : 'rgb(205, 50, 50)') . ';
                    color: white; font-weight: bold; border-radius: 30px; font-size: 14px;">
                    ' . ($slide['status'] === 'Publish' ? 'Published' : 'Unpublished') . '
                </button>
                </form>';

            $actionButtons = '
                <button class="btn btn-primary btn-edit-carousel" data-bs-toggle="modal" data-bs-target="#editModal"
                    ' . $dataAttributes . '>
                    <i class="fa fa-edit"></i>
                </button>
                <button class="btn btn-secondary btn-delete-carousel" data-id="' . esc($slide['sno']) . '"
                    data-title="' . esc($slide['title']) . '" data-bs-toggle="modal" data-bs-target="#deleteCarousel">
                    <i class="fa fa-trash"></i>
                </button>';

                $data[] = [
                    'carousel_image' => base_url('uploads/Carousel/' . $slide['image']),
                    'title' => esc($slide['title']),
                    'subtitle' => esc($slide['sub-title']),
                    'created_at' => date('d M Y', strtotime($slide['created_at'])),
                    'description' => esc($slide['description']),
                    'status' => $statusButton,
                    'action' => $actionButtons,
                ];
            }

            return $this->response->setJSON([
                'draw' => (int) $request->getGet('draw'),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $filteredRecords,
                'data' => $data,
                'order' => [
                    'column' => $orderColumnIndex,
                    'dir' => $orderDir
                ]
            ]);
    }

    public function delete($id = null)
    {
        log_message('debug', 'Delete function called with ID: ' . $id);

        if ($this->request->isAJAX()) {
            if ($id === null || !is_numeric($id)) {
                return $this->response->setJSON(['status' => 'error', 'message' => 'Invalid ID'])->setStatusCode(400);
            }

            $deleted = $this->carouselModel->protect(false)->update($id, [
                'deleted_at' => date('Y-m-d H:i:s')
            ]);

            if ($deleted) {
                return $this->response->setJSON(['status' => 'success', 'message' => 'Carousel deleted successfully']);
            } else {
                return $this->response->setJSON(['status' => 'error', 'message' => 'Failed to delete carousel']);
            }
        }

        return redirect()->back()->with('error', 'Invalid request');
    }

    public function store()
    {
        $file = $this->request->getFile('image');

        if (!$file || !$file->isValid() || $file->hasMoved()) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Invalid image file.',
            ]);
        }

        $newName = $file->getRandomName();
        $file->move(FCPATH . 'uploads/Carousel', $newName);

        $data = [
            'title' => $this->request->getPost('title'),
            'sub-title' => $this->request->getPost('sub-title'),
            'date' => date('Y-m-d'),
            'description' => $this->request->getPost('description'),
            'status' => $this->request->getPost('status') ? 'Publish' : 'Unpublish',
            'image' => $newName,
        ];

        $this->carouselModel->save($data);

        // Detect AJAX request
        if ($this->request->isAJAX()) {
            return $this->response->setJSON([
                'status' => 'success',
                'message' => 'Carousel added successfully.',
            ]);
        }

        return redirect()->to('/admin/carousel')->with('success', 'Slide added!');
    }

    public function edit($id)
    {
        $carouselModel = new \App\Models\CarouselModel();
        $slide = $carouselModel->find($id);

        if (!$slide) {
            return redirect()->to('admin/carousel')->with('error', 'Slide not found.');
        }

        return view('admin/carousel/edit', ['slide' => $slide]);
    }

    public function update()
    {
        $sno = $this->request->getPost('sno');
        // $sno = $this->request->getPost('sno');
        if (!$sno) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'SNO not received.'
            ]);
        }

        $data = [
            'title' => $this->request->getPost('title'),
            'sub-title' => $this->request->getPost('sub-title'),
            'description' => $this->request->getPost('description'),
            'status' => $this->request->getPost('status') ? 'Publish' : 'Unpublish',
            'updated_at' => date('Y-m-d H:i:s'),
        ];

        // Handle image
        $image = $this->request->getFile('image');
        if ($image && $image->isValid() && !$image->hasMoved()) {
            $newName = $image->getRandomName();
            $image->move('uploads/Carousel', $newName);
            $data['image'] = $newName;
        }

        $this->carouselModel->update($sno, $data);

        if ($this->request->isAJAX()) {
            return $this->response->setJSON([
                'status' => 'success',
                'message' => 'Carousel updated successfully.',
            ]);
        }

        return redirect()->to(base_url('admin/carousel/'))->with('success', 'Carousel updated.');
    }

    public function dashboard()
    {
        return view('admin/dashboard');
    }

    public function toggleStatus($type, $id)
    {
        $modelMap = [
            'blog'     => \App\Models\BlogModel::class,
            'carousel' => \App\Models\CarouselModel::class,
            'event' => \App\Models\EventModel::class,
        ];

        if (!isset($modelMap[$type])) {
            return redirect()->back()->with('error', 'Invalid type');
        }

        $model = new $modelMap[$type]();
        $item = $model->find($id);

        if (!$item) {
            return redirect()->back()->with('error', 'Item not found');
        }

        $newStatus = ($item['status'] === 'Publish') ? 'Unpublish' : 'Publish';
        $model->update($id, ['status' => $newStatus]);

        return redirect()->back();
    }

    public function blog()
    {
        if ($this->request->isAJAX()) {
            $request = service('request');

            $searchValue = $request->getGet('search')['value'] ?? '';
            $searchDate = $request->getGet('searchDate') ?? '';
            $status = $request->getGet('status') ?? '';

            $page = (int) ($this->request->getGet('page') ?? 1);
            $length = (int) $request->getGet('length') ?: 10;
            $start = ($page - 1) * $length;

            $model = new \App\Models\BlogModel();
            $builder = $model->builder()->where('deleted_at', null);

            // Filters
            if ($searchValue) {
                $builder->like('title', $searchValue);
            }

            if ($searchDate) {
                $builder->where('date', $searchDate);
            }

            if ($status && in_array($status, ['Publish', 'Unpublish'])) {
                $builder->where('status', $status);
            }

            $filteredBuilder = clone $builder;
            $recordsFiltered = $filteredBuilder->countAllResults();

            // Sorting
            $orderColumnIndex = $request->getGet('order')[0]['column'] ?? 3;
            $orderDir = $request->getGet('order')[0]['dir'] ?? 'desc';
            $columnsMap = [
                1 => 'title',
                3 => 'date',
            ];
            $orderColumn = $columnsMap[$orderColumnIndex] ?? 'created_at';
            $builder->orderBy($orderColumn, $orderDir);

            // Fetch paginated results
            $blogs = $builder->limit($length, $start)->get()->getResultArray();

            // Format
            $data = [];
            foreach ($blogs as $index => $row) {
                $data[] = [
                    'sno' => $start + $index + 1,
                    'title' => esc($row['title']),
                    'image' => '<img src="' . base_url('uploads/Blog/' . $row['image']) . '" style="max-width: 100px; max-height: 100px;">',
                    'date' => date('d M Y', strtotime($row['date'])),
                    'status' => '<form action="' . base_url('admin/toggleStatus/blog/' . $row['sno']) . '" method="post">
                                    <button type="submit" class="btn w-55"
                                        style="background-color: ' . ($row['status'] === 'Publish' ? '#32CD32' : 'rgb(205, 50, 50)') . ';
                                        color: white; font-weight: bold; border-radius: 30px; font-size: 14px;">
                                        ' . $row['status'] . '
                                    </button>
                                </form>',
                    'action' => '<a href="' . base_url('admin/blog/editBlog/' . $row['sno']) . '" class="btn btn-primary"><i class="fa fa-edit"></i></a>
                                <button class="btn btn-secondary btn-delete-blog" data-id="' . $row['sno'] . '" data-title="' . esc($row['title']) . '" data-bs-toggle="modal" data-bs-target="#deleteBlogpost">
                                    <i class="fa fa-trash"></i>
                                </button>'
                ];
            }

            // Total records (not filtered)
            $recordsTotal = $model->where('deleted_at', null)->countAllResults();

            return $this->response->setJSON([
                'draw' => (int) $request->getGet('draw'),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => $data,
                'order' => [
                    'column' => $orderColumnIndex,
                    'dir' => $orderDir
                ]
            ]);
        }

        return view('admin/blog/blog-post');
    }

    public function storeBlog()
    {
        $file = $this->request->getFile('image');


        if (!$file) {
            dd('No file selected.');
        }

        if (!$file->isValid()) {
            dd('File is not valid.', $file->getErrorString(), $file->getError());
        }

        if ($file->hasMoved()) {
            dd('File has already been moved.');
        }

        $newName = $file->getRandomName();

        $moved = $file->move(FCPATH . 'uploads/Blog', $newName);

        if (!$moved) {
            dd('File failed to move.');
        }

        $this->blogModel->save([
            'title' => $this->request->getPost('title'),
            'date' => date('Y-m-d'),
            'description' => $this->request->getPost('description'),
            'status' => $this->request->getPost('status') ? 'Publish' : 'Unpublish',
            'image' => $newName,
        ]);

        return redirect()->to('/admin/blog/blog-post')->with('success', 'Blog added!');
    }

    public function createBlog(){
        return view('admin/blog/createBlog');
    }

    public function editBlog($id){
        $model = new \App\Models\BlogModel();
        $post = $model->find($id);

        if (!$post) {
            return redirect()->back()->with('error', 'Blog not found.');
        }
        return view('admin/blog/editBlog', ['post' => $post]);
    }

    public function updateBlog()
    {
        $sno = $this->request->getPost('sno');  

        if (!$sno) {
            return redirect()->back()->with('error', 'Invalid blog post ID.');
        }

        $data = [
            'title' => $this->request->getPost('title'),
            'date' => $this->request->getPost('date'),
            'description' => $this->request->getPost('description'),
            'status' => $this->request->getPost('status') ? 'Publish' : 'Unpublish',
            'updated_at' => date('Y-m-d H:i:s'),
        ];

        $image = $this->request->getFile('image');
        if ($image && $image->isValid() && !$image->hasMoved()) {
            $newName = $image->getRandomName();
            $image->move('uploads/Blog', $newName);
            $data['image'] = $newName;
        }

        $model = new \App\Models\BlogModel();
        $model->update($sno, $data);

        return redirect()->to(base_url('admin/blog/blog-post'))->with('success', 'Blog updated successfully.');
    }

    public function deleteBlog($id)
    {
        $post = $this->blogModel->find($id);
        if (!$post) {
            return redirect()->back()->with('error', 'Blog not found.');
        }

        $this->blogModel->update($id, [
            'deleted_at' => date('Y-m-d H:i:s'),
        ]);
        $this->blogModel->delete($id);
        return redirect()->to('/admin/blog/blog-post')->with('success', 'Blog deleted!');
    }

    // event
    public function event()
    {
        if ($this->request->isAJAX()) {
            $request = service('request');

            $searchValue = $request->getGet('search')['value'] ?? '';
            $searchDate = $request->getGet('searchDate') ?? '';
            $status = $request->getGet('status') ?? '';

            $page = (int) ($this->request->getGet('page') ?? 1);
            $length = (int) $request->getGet('length') ?: 10;
            $start = ($page - 1) * $length;

            $model = new \App\Models\EventModel();
            $builder = $model->builder()->where('deleted_at', null);

            if ($searchValue) {
                $builder->like('title', $searchValue);
            }

            if ($searchDate) {
                $builder->where('date', $searchDate);
            }

            if ($status && in_array($status, ['Publish', 'Unpublish'])) {
                $builder->where('status', $status);
            }

            $filteredBuilder = clone $builder;
            $recordsFiltered = $filteredBuilder->countAllResults();

            $orderColumnIndex = $request->getGet('order')[0]['column'] ?? 3;
            $orderDir = $request->getGet('order')[0]['dir'] ?? 'desc';
            $columnsMap = [
                1 => 'title',
                4 => 'date',
            ];
            $orderColumn = $columnsMap[$orderColumnIndex] ?? 'created_at';
            $builder->orderBy($orderColumn, $orderDir);

            $events = $builder->limit($length, $start)->get()->getResultArray();

            $data = [];
            foreach ($events as $index => $row) {
                $data[] = [
                    'sno' => $start + $index + 1,
                    'title' => esc($row['title']),
                    'image' => '<img src="' . base_url('uploads/Event/' . $row['image']) . '" style="max-width: 100px; max-height: 100px;">',
                    'introduction' => esc($row['introduction']),
                    'date' => date('d M Y', strtotime($row['date'])),
                    'address' => esc($row['address']),
                    'status' => '<form action="' . base_url('admin/toggleStatus/event/' . $row['sno']) . '" method="post">
                                    <button type="submit" class="btn w-55"
                                        style="background-color: ' . ($row['status'] === 'Publish' ? '#32CD32' : 'rgb(205, 50, 50)') . ';
                                        color: white; font-weight: bold; border-radius: 30px; font-size: 14px;">
                                        ' . $row['status'] . '
                                    </button>
                                </form>',
                    'action' => '<a href="' . base_url('admin/event/editEvent/' . $row['sno']) . '" class="btn btn-primary"><i class="fa fa-edit"></i></a>
                                <button class="btn btn-secondary btn-delete-event" 
                                    data-id="' . $row['sno'] . '" 
                                    data-title="' . esc($row['title']) . '" 
                                    data-bs-toggle="modal" 
                                    data-bs-target="#deleteEvent">
                                    <i class="fa fa-trash"></i>
                                </button>'
                ];
            }

            $recordsTotal = $model->where('deleted_at', null)->countAllResults();

            return $this->response->setJSON([
                'draw' => (int) $request->getGet('draw'),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => $data,
                'order' => [
                    'column' => $orderColumnIndex,
                    'dir' => $orderDir
                ]
            ]);
        }

        return view('admin/event/manage-event');
    }

    public function storeEvent()
    {
        $file = $this->request->getFile('image');


        if (!$file) {
            dd('No file selected.');
        }

        if (!$file->isValid()) {
            dd('File is not valid.', $file->getErrorString(), $file->getError());
        }

        if ($file->hasMoved()) {
            dd('File has already been moved.');
        }

        $newName = $file->getRandomName();

        $moved = $file->move(FCPATH . 'uploads/Event', $newName);

        if (!$moved) {
            dd('File failed to move.');
        }


        $this->eventModel->save([
            'title' => $this->request->getPost('title'),
            'date' => date('Y-m-d'),
            'introduction' => $this->request->getPost('introduction'),
            'description' => $this->request->getPost('description'),
            'address' => $this->request->getPost('address'),
            'status' => $this->request->getPost('status') ? 'Publish' : 'Unpublish',
            'image' => $newName,
        ]);

        return redirect()->to('/admin/event/manage-event')->with('success', 'Event added!');
    }

    public function createEvent(){
        return view('admin/event/createEvent');
    }

    public function editEvent($id)
    {
        $model = new \App\Models\EventModel();
        $event = $model->find($id);

        if (!$event) {
            return redirect()->back()->with('error', 'Event not found.');
        }
        return view('admin/event/editEvent', ['event' => $event]);
    }

    public function updateEvent()
    {
        $sno = $this->request->getPost('sno');  

        if (!$sno) {
            return redirect()->back()->with('error', 'Invalid Event ID.');
        }

        $data = [
            'title' => $this->request->getPost('title'),
            'date' => $this->request->getPost('date'),
            'introduction' => $this->request->getPost('introduction'),
            'address' => $this->request->getPost('address'),
            'description' => $this->request->getPost('description'),
            'status' => $this->request->getPost('status') ? 'Publish' : 'Unpublish',
            'updated_at' => date('Y-m-d H:i:s'),
        ];

        $image = $this->request->getFile('image');
        if ($image && $image->isValid() && !$image->hasMoved()) {
            $newName = $image->getRandomName();
            $image->move('uploads/Event', $newName);
            $data['image'] = $newName;
        }

        $model = new \App\Models\EventModel();
        $model->update($sno, $data);

        return redirect()->to(base_url('admin/event/manage-event'))->with('success', 'Event updated successfully.');
    }

    // public function deleteEvent($id)
    // {
    //     $this->eventModel->update($id, [
    //         'deleted_at' => date('Y-m-d H:i:s'),
    //     ]);
    //     $this->eventModel->delete($id);
    //     return redirect()->to('/admin/event/manage-event')->with('success', 'Event deleted!');
    // }

    public function deleteEvent()
    {
        if ($this->request->isAJAX()) {
            $data = json_decode($this->request->getBody(), true);
            $id = $data['id'] ?? null;

            if (!$id) {
                return $this->response->setJSON(['success' => false, 'message' => 'Invalid event ID']);
            }

            $eventModel = new \App\Models\EventModel();
            
            if ($eventModel->delete($id)) {
                return $this->response->setJSON(['success' => true]);
            } else {
                return $this->response->setJSON(['success' => false, 'message' => 'Could not delete event']);
            }
        }

        return redirect()->to('admin/event/manage-event');
    }

    // members
    public function members()
    {
        if ($this->request->isAJAX()) {
            $request = service('request');

            $searchValue = $request->getGet('search')['value'] ?? '';
            $searchDate = $request->getGet('searchDate') ?? '';

            $page = (int) ($request->getGet('page') ?? 1);
            $length = (int) $request->getGet('length') ?: 10;
            $start = ($page - 1) * $length;

            $model = new \App\Models\MemberModel();
            $builder = $model->builder()
                        ->select('sno, first_name, last_name, date, email, phone, address')
                        ->where('deleted_at', null);

            // Filters
            if ($searchValue) {
                $builder->groupStart()
                        ->like('first_name', $searchValue)
                        ->orLike('last_name', $searchValue)
                        ->groupEnd();
            }

            if ($searchDate) {
                $builder->where('date', $searchDate);
            }

            $filteredBuilder = clone $builder;
            $recordsFiltered = $filteredBuilder->countAllResults();

            // Sorting
            $orderColumnIndex = $request->getGet('order')[0]['column'] ?? 3;
            $orderDir = $request->getGet('order')[0]['dir'] ?? 'desc';
            $columnsMap = [
                1 => 'first_name',
                2 => 'last_name',
                3 => 'date',
            ];
            $orderColumn = $columnsMap[$orderColumnIndex] ?? 'created_at';
            $builder->orderBy($orderColumn, $orderDir);

            // Fetch paginated results
            $memberItems = $builder->limit($length, $start)->get()->getResultArray();

            // Format
            $data = [];
            foreach ($memberItems as $index => $row) {

                $data[] = [
                    'sno'      => $start + $index + 1,
                    'first_name'     => esc($row['first_name']),
                    'last_name'     => esc($row['last_name']),
                    'date'     => date('d M Y', strtotime($row['date'])),
                    'email' => esc($row['email']),
                    'phone' => esc($row['phone']),
                    'address' => esc($row['address']),
                    'action'   => '
                        <button class="btn btn-primary btn-edit-member"
                            data-id="' . esc($row['sno']) . '"
                            data-firstname="' . esc($row['first_name']) . '"
                            data-lastname="' . esc($row['last_name']) . '"
                            data-email="' . esc($row['email']) . '"
                            data-phone="' . esc($row['phone']) . '"
                            data-address="' . esc($row['address']) . '"
                            data-date="' . esc($row['date']) . '"
                            data-bs-toggle="modal" data-bs-target="#exampleModal">
                            <i class="fa fa-edit"></i>
                        </button>
                        <button class="btn btn-secondary btn-delete-member"
                            data-id="' . esc($row['sno']) . '"
                            data-title="' . esc($row['first_name'] . ' ' . $row['last_name']) . '"
                            data-bs-toggle="modal" data-bs-target="#deleteMember">
                            <i class="fa fa-trash"></i>
                        </button>'
                ];
            }

            $recordsTotal = $model->where('deleted_at', null)->countAllResults();

            return $this->response->setJSON([
                'draw' => (int) $request->getGet('draw'),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => $data,
                'order' => [
                    'column' => $orderColumnIndex,
                    'dir' => $orderDir
                ]
            ]);
        }

        return view('admin/manage-members');
    }

    public function updateMember()
    {
        $id = $this->request->getPost('sno'); 

        if (!$id) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Invalid Member ID.',
            ]);
        }

        $data = [
            'first_name' => $this->request->getPost('first_name'),
            'last_name' => $this->request->getPost('last_name'),
            'date' => $this->request->getPost('date'),
            'address' => $this->request->getPost('address'),
            'phone' => $this->request->getPost('phone'),
            'email' => $this->request->getPost('email'),
            'updated_at' => date('Y-m-d H:i:s'),
        ];

        $model = new \App\Models\MemberModel();
        $model->update($id, $data);

        return $this->response->setJSON([
            'status' => 'success',
            'message' => 'Member updated successfully.',
        ]);
    }

    public function deleteMember($id)
    {
        try {
            $this->memberModel->update($id, ['deleted_at' => date('Y-m-d H:i:s')]);
            $this->memberModel->delete($id);

            return $this->response->setJSON(['status' => 'success', 'message' => 'Member deleted successfully.']);
        } catch (\Exception $e) {
            return $this->response->setJSON(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    // users
    public function users()
    {
        if ($this->request->isAJAX()) {
            $request = service('request');

            $searchValue = $request->getGet('search')['value'];
            $searchDate = $request->getGet('searchDate') ?? '';

            $page = (int) ($request->getGet('page') ?? 1);
            $length = (int) $request->getGet('length') ?: 10;
            $start = ($page - 1) * $length;

            $model = new \App\Models\UserModel();
            $builder = $model->builder()->where('deleted_at', null);

            // Filters
            if ($searchValue) {
                $builder->groupStart()
                ->like('name', $searchValue)
                ->orLike('email', $searchValue)
                ->groupEnd();
            }

            if ($searchDate) {
                $builder->where('date', $searchDate);
            }

            $filteredBuilder = clone $builder;
            $recordsFiltered = $filteredBuilder->countAllResults();

            // Sorting
            $orderColumnIndex = $request->getGet('order')[0]['column'] ?? 3;
            $orderDir = $request->getGet('order')[0]['dir'] ?? 'desc';
            $columnsMap = [
                1 => 'name',
                2 => 'email',
            ];
            $orderColumn = $columnsMap[$orderColumnIndex] ?? 'created_at';
            $builder->orderBy($orderColumn, $orderDir);

            // Fetch paginated results
            $userItems = $builder->limit($length, $start)->get()->getResultArray();

            // Format
            $data = [];
            foreach ($userItems as $index => $row) {

                $data[] = [
                    'sno'      => $start + $index + 1,
                    'name'     => esc($row['name']),
                    'email' => esc($row['email']),
                    'phone' => esc($row['phone']),
                    'subject' => esc($row['subject']),
                    'message' => esc($row['message']),
                    'action'   => '
                        <button class="btn btn-secondary btn-delete-user"
                            data-id="' . esc($row['sno']) . '"
                            data-title="' . esc($row['name']) .'"
                            data-bs-toggle="modal" data-bs-target="#deleteUser">
                            <i class="fa fa-trash"></i>
                        </button>'
                ];
            }

            $recordsTotal = $model->where('deleted_at', null)->countAllResults();

            return $this->response->setJSON([
                'draw' => (int) $request->getGet('draw'),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => $data,
                'order' => [
                    'column' => $orderColumnIndex,
                    'dir' => $orderDir
                ]
            ]);
        }

        return view('admin/manage-users');
    }

    public function deleteUser()
    {
        if ($this->request->isAJAX()) {
            $data = json_decode($this->request->getBody(), true);
            $id = $data['id'] ?? null;

            if (!$id) {
                return $this->response->setJSON(['success' => false, 'message' => 'Invalid user ID']);
            }

            $userModel = new \App\Models\UserModel();
            
            if ($userModel->delete($id)) {
                return $this->response->setJSON(['success' => true]);
            } else {
                return $this->response->setJSON(['success' => false, 'message' => 'Could not delete user']);
            }
        }

        return redirect()->to('admin/manage-users');
    }

    // gallery
    public function gallery()
    {
        if ($this->request->isAJAX()) {
            $request = service('request');

            $searchValue = $request->getGet('search')['value'] ?? '';
            $searchDate = $request->getGet('searchDate') ?? '';

            $page = (int) ($request->getGet('page') ?? 1);
            $length = (int) $request->getGet('length') ?: 10;
            $start = ($page - 1) * $length;

            $model = new \App\Models\GalleryModel();
            $builder = $model->builder()->where('deleted_at', null);

            // Filters
            if ($searchValue) {
                $builder->groupStart()
                        ->like('name', $searchValue)
                        ->orLike('category', $searchValue)
                        ->groupEnd();
            }

            if ($searchDate) {
                $builder->where('date', $searchDate);
            }

            $filteredBuilder = clone $builder;
            $recordsFiltered = $filteredBuilder->countAllResults();

            // Sorting
            $orderColumnIndex = $request->getGet('order')[0]['column'] ?? 3;
            $orderDir = $request->getGet('order')[0]['dir'] ?? 'desc';
            $columnsMap = [
                1 => 'name',
                2 => 'category',
                3 => 'date',
            ];
            $orderColumn = $columnsMap[$orderColumnIndex] ?? 'created_at';
            $builder->orderBy($orderColumn, $orderDir);

            // Fetch paginated results
            $galleryItems = $builder->limit($length, $start)->get()->getResultArray();

            // Format
            $data = [];
            foreach ($galleryItems as $index => $row) {
                $images = json_decode($row['image'], true);
                $imageTag = '';
                if (!empty($images)) {
                    $imageTag = '<img src="' . base_url('uploads/gallery/' . $images[0]) . '" style="max-width: 100px; max-height: 100px;">';
                }

                $data[] = [
                    'sno'      => $start + $index + 1,
                    'name'     => esc($row['name']),
                    'category' => esc($row['category']),
                    'date'     => date('d M Y', strtotime($row['date'])),
                    'image'    => $imageTag,
                    'action'   => '
                        <button class="btn btn-warning view-gallery-btn"
                            data-id="' . esc($row['sno']) . '"
                            data-title="' . esc($row['name']) . '"
                            data-bs-toggle="modal" data-bs-target="#galleryModal">
                            <i class="fa fa-eye"></i>
                        </button>
                        <button class="btn btn-primary edit-gallery-btn"
                            data-id="' . esc($row['sno']) . '"
                            data-name="' . esc($row['name']) . '"
                            data-category="' . esc($row['category']) . '"
                            data-date="' . esc($row['date']) . '"
                            data-bs-toggle="modal" data-bs-target="#editModal">
                            <i class="fa fa-edit"></i>
                        </button>
                        <button class="btn btn-secondary btn-delete-gallery"
                            data-id="' . esc($row['sno']) . '"
                            data-title="' . esc($row['name']) . '"
                            data-bs-toggle="modal" data-bs-target="#deleteGallery">
                            <i class="fa fa-trash"></i>
                        </button>'
                ];
            }

            $recordsTotal = $model->where('deleted_at', null)->countAllResults();

            return $this->response->setJSON([
                'draw' => (int) $request->getGet('draw'),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsFiltered,
                'data' => $data,
                'order' => [
                    'column' => $orderColumnIndex,
                    'dir' => $orderDir
                ]
            ]);
        }

        return view('admin/manage-gallery');
    }

    public function fetchGalleryImages()
    {
        if ($this->request->isAJAX()) {
            $sno = $this->request->getPost('id');
            $gallery = $this->galleryModel->find($sno);

            if (!$gallery) {
                return $this->response->setJSON(['status' => 'error', 'message' => 'Gallery not found.']);
            }

            $images = json_decode($gallery['image'], true);
            $html = '';

            foreach ($images as $img) {
                $imgPath = base_url('uploads/gallery/' . $img);
                $html .= '
                    <div class="col-md-4 mb-3">
                        <img style="aspect-ratio:5/3;border-radius:10px;" src="' . $imgPath . '" alt="' . esc($img) . '" class="img-fluid">
                        <button style="border-radius:10px" class="btn w-100 btn-outline-danger btn-sm mt-2 delete-image-btn"
                            data-id="' . esc($gallery['sno']) . '"
                            data-img="' . esc($img) . '"
                            data-title="' . esc($gallery['name']) . '">
                            Delete
                        </button>
                    </div>';
            }

            return $this->response->setJSON([
                'status' => 'success',
                'title' => esc($gallery['name']),
                'images' => $html
            ]);
        }
    }

    public function deleteGalleryImage()
    {
        if ($this->request->isAJAX()) {
            $id = $this->request->getPost('id');
            $img = $this->request->getPost('img');

            $gallery = $this->galleryModel->find($id);
            if (!$gallery) {
                return $this->response->setJSON(['status' => 'error', 'message' => 'Gallery not found']);
            }

            $images = json_decode($gallery['image'], true);
            if (($key = array_search($img, $images)) !== false) {
                unset($images[$key]);
                $this->galleryModel->update($id, [
                    'image' => json_encode(array_values($images))
                ]);
                @unlink(FCPATH . 'uploads/gallery/' . $img);
            }

            return $this->response->setJSON(['status' => 'success', 'message' => 'Image deleted']);
        }
    }

    public function storeGallery()
    {
        $name = $this->request->getPost('name');
        $category = $this->request->getPost('category');
        $date = $this->request->getPost('date');

        $imageNames = [];

        $files = $this->request->getFiles();
        if ($files && isset($files['image'])) {
            foreach ($files['image'] as $file) {
                if ($file->isValid() && !$file->hasMoved()) {
                    $newName = $file->getRandomName();
                    $file->move(FCPATH . 'uploads/gallery', $newName);
                    $imageNames[] = $newName;
                }
            }
        }

        $this->galleryModel->save([
            'name' => $name,
            'category' => $category,
            'date' => $date,
            'image' => json_encode($imageNames), // Save as JSON string
        ]);
        return redirect()->to('/admin/manage-gallery')->with('success', 'Slide added!');
    }

    public function updateGallery()
    {
        $sno = $this->request->getPost('sno');

        if (!$sno) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => 'Invalid gallery ID.'
            ]);
        }

        $data = [
            'name' => $this->request->getPost('name'),
            'date' => $this->request->getPost('date'),
            'category' => $this->request->getPost('category'),
            'updated_at' => date('Y-m-d H:i:s'),
        ];

        // Handle image uploads
        $uploadedImages = [];
        $images = $this->request->getFiles();

        if (isset($images['image'])) {
            foreach ($images['image'] as $img) {
                if ($img->isValid() && !$img->hasMoved()) {
                    $newName = $img->getRandomName();
                    $img->move('uploads/gallery', $newName);
                    $uploadedImages[] = $newName;
                }
            }

            if (!empty($uploadedImages)) {
                $data['image'] = json_encode($uploadedImages);
            }
        }

        try {
            $model = new \App\Models\GalleryModel();
            $model->update($sno, $data);

            return $this->response->setJSON([
                'status' => 'success',
                'message' => 'Gallery updated successfully.'
            ]);
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    public function deleteGallery()
    {
        if ($this->request->isAJAX()) {
            $data = json_decode($this->request->getBody(), true);
            $id = $data['id'] ?? null;

            if (!$id) {
                return $this->response->setJSON(['success' => false, 'message' => 'Invalid gallery ID']);
            }

            // Load your Gallery model
            $galleryModel = new \App\Models\GalleryModel();

            // Delete gallery by id
            if ($galleryModel->delete($id)) {
                return $this->response->setJSON(['success' => true]);
            } else {
                return $this->response->setJSON(['success' => false, 'message' => 'Could not delete gallery']);
            }
        }

        return redirect()->to('/admin/gallery');
    }

    // public function profile()
    // {

    //     // dd(session()->get());

    //     $profileId = session()->get('id');

    //     if (!$profileId) {
    //         return redirect()->to('admin/profile')->with('error', 'Please log in to continue.');
    //     }

    //     $profile = $this->profileModel->find($profileId);

    //     if (!$profile) {
    //         return redirect()->back()->with('error', 'Admin not found.');
    //     }

    //     return view('admin/profile', ['profile' => $profile]);
    // }

    public function profile()
    {
        $profileId = session()->get('id');

        if (!$profileId) {
            return redirect()->to('admin/login')->with('error', 'Please log in to continue.');
        }

        $profile = $this->profileModel->find($profileId);

        if (!$profile) {
            return redirect()->back()->with('error', 'Admin not found.');
        }

        return view('admin/profile', ['profile' => $profile]);
    }

    public function updateProfile()
    {
        $validation = \Config\Services::validation();

        $rules = [
            'fullname' => 'required',
            'email'    => 'required|valid_email',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        $profileId = session()->get('id');

        $data = [
            'fullname' => $this->request->getPost('fullname'),
            'email'    => $this->request->getPost('email'),
        ];

        $profileModel = new \App\Models\ProfileModel(); 
        $profileModel->update($profileId, $data);

        return redirect()->to('admin/profile')->with('success', 'Profile updated successfully');
    }

    public function changePassword()
    {
        $validation = \Config\Services::validation();

        $rules = [
            'password'       => 'required',
            'newpassword'    => 'required',
            'renewpassword'  => 'required|matches[newpassword]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('errors', $validation->getErrors());
        }

        $profileId = session()->get('id');
        $profileModel = new \App\Models\ProfileModel();
        $profile = $profileModel->find($profileId);

        if (!password_verify($this->request->getPost('password'), $profile['password'])) {
            return redirect()->back()->with('error', 'Current password is incorrect.');
        }

        $newPassword = password_hash($this->request->getPost('newpassword'), PASSWORD_DEFAULT);

        $profileModel->update($profileId, ['password' => $newPassword]);

        return redirect()->back()->with('success', 'Password changed successfully!');
    }



    // public function updateProfile()
    // {
    //     $validation = \Config\Services::validation();

    //     $rules = [
    //         'fullname' => 'required|min_length[10]',
    //         'email'    => 'required|valid_email',
    //     ];

    //     if (!$this->validate($rules)) {
    //         return redirect()->back()->withInput()->with('errors', $validation->getErrors());
    //     }

    //     $profileId = session()->get('id');

    //     $data = [
    //         'fullname' => $this->request->getPost('fullname'),
    //         'email'    => $this->request->getPost('email'),
    //     ];

    //     $profileModel = new \App\Models\ProfileModel(); 
    //     $profileModel->update($profileId, $data);

    //     // ✅ Set flashdata and redirect — DO NOT dd() here
    //     // session()->setFlashdata('success', 'Profile updated successfully.');
    //     return redirect()->to('admin/profile')->with('success','Profile updated successfully');
    // }

    // public function changePassword()
    // {
    //     $validation = \Config\Services::validation();

    //     $rules = [
    //         'password'       => 'required',
    //         'newpassword'    => 'required|min_length[6]',
    //         'renewpassword'  => 'required|matches[newpassword]',
    //     ];

    //     if (!$this->validate($rules)) {
    //         return redirect()->back()->withInput()->with('errors', $validation->getErrors());
    //     }

    //     $profileId = session()->get('id');
    //     $profileModel = new \App\Models\ProfileModel();
    //     $profile = $profileModel->find($profileId);

    //     // Validate current password
    //     if (!password_verify($this->request->getPost('password'), $profile['password'])) {
    //         return redirect()->back()->with('error', 'Current password is incorrect.');
    //     }

    //     // Update to new password
    //     $newPassword = password_hash($this->request->getPost('newpassword'), PASSWORD_DEFAULT);

    //     $profileModel->update($profileId, ['password' => $newPassword]);

    //     return redirect()->back()->with('success', 'Password changed successfully!');
    // }

    public function login()
    {
        return view('admin/login');
    }


    public function handleLogin()
    {
        $request = service('request');
        $email = $request->getPost('email');
        $password = $request->getPost('password');

        if (!$email || !$password) {
            return redirect()->back()->with('login_error', 'Please enter both email and password.')->withInput();
        }

        $profile = $this->profileModel->where('email', $email)->first();

        if ($profile && $password === $profile['password']) {
            session()->set([
                'id' => $profile['id'],
                'fullname' => $profile['fullname'],
                'email' => $profile['email'],
                'isLoggedIn' => true,
            ]);
            return redirect()->to('admin/dashboard');
        } else {
            return redirect()->back()->with('login_error', 'Invalid email or password.')->withInput();
        }
    }

    
}
